<?php
require_once 'config.php';

// Check if user is logged in
$user_info = getUserInfo($client, $google_oauth);
if (!$user_info) {
    header('Location: login.php');
    exit();
}

$user_calculations = getUserZakatHistory($_SESSION['user_email']);
$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $calculation_id = $_POST['calculation_id'] ?: null;
    $payment_amount = floatval($_POST['payment_amount'] ?? 0);
    $payment_date = $_POST['payment_date'] ?? date('Y-m-d');
    $payment_method = $_POST['payment_method'] ?? '';
    $description = $_POST['description'] ?? '';
    
    // Validate payment amount
    if ($payment_amount <= 0) {
        $error = "Payment amount must be greater than 0.";
    } else {
        $payment_data = [
            'calculation_id' => $calculation_id,
            'payment_amount' => $payment_amount,
            'payment_date' => $payment_date,
            'payment_method' => $payment_method,
            'description' => $description
        ];
        
        $payment_id = addZakatPayment($_SESSION['user_email'], $payment_data);
        
        if ($payment_id) {
            $success = "Payment recorded successfully!";
            // Clear form
            $_POST = [];
        } else {
            $error = "Failed to record payment. Please try again.";
        }
    }
}

// Get Zakat summary
$zakat_summary = getUserZakatSummary($_SESSION['user_email']);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Payment - Mominist</title>
    <style>
        body {
            padding-top: 80px;
            font-family: Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            margin: 0;
            min-height: 100vh;
        }

        .payment-container {
            max-width: 800px;
            margin: 20px auto;
            padding: 0 20px;
        }

        .payment-card {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
            margin-bottom: 20px;
        }

        .summary-card {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #333;
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
        }

        .form-group textarea {
            height: 100px;
            resize: vertical;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            font-weight: bold;
            text-decoration: none;
            display: inline-block;
            text-align: center;
            transition: all 0.3s;
        }

        .btn-success {
            background: #28a745;
            color: white;
        }

        .btn-primary {
            background: #4285f4;
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn:hover {
            opacity: 0.9;
            transform: translateY(-2px);
        }

        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap;
        }

        .alert {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .summary-item {
            margin: 10px 0;
            font-size: 1.1em;
        }

        .summary-amount {
            font-size: 1.3em;
            font-weight: bold;
            margin: 5px 0;
        }

        .remaining-warning {
            color: #ffc107;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <!-- Include Navbar -->
    <?php include 'navbar.php'; ?>

    <div class="payment-container">
        <!-- Zakat Summary -->
        <?php if ($zakat_summary): ?>
        <div class="summary-card">
            <h2>💰 Your Zakat Summary</h2>
            <div class="summary-item">
                Total Zakat Calculated: 
                <div class="summary-amount">Rs <?php echo number_format($zakat_summary['total_zakat'], 2); ?></div>
            </div>
            <div class="summary-item">
                Total Paid: 
                <div class="summary-amount">Rs <?php echo number_format($zakat_summary['total_paid'], 2); ?></div>
            </div>
            <div class="summary-item">
                Remaining Zakat: 
                <div class="summary-amount <?php echo $zakat_summary['remaining_zakat'] > 0 ? 'remaining-warning' : ''; ?>">
                    Rs <?php echo number_format($zakat_summary['remaining_zakat'], 2); ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <div class="payment-card">
            <h1 style="text-align: center; color: #333; margin-bottom: 30px;">💳 Record Zakat Payment</h1>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <?php echo $success; ?>
                </div>
            <?php endif; ?>

            <form method="POST" action="">
                <div class="form-group">
                    <label for="calculation_id">Link to Calculation (Optional):</label>
                    <select id="calculation_id" name="calculation_id">
                        <option value="">-- Select Calculation --</option>
                        <?php foreach ($user_calculations as $calc): ?>
                            <option value="<?php echo $calc['id']; ?>" <?php echo ($_POST['calculation_id'] ?? '') == $calc['id'] ? 'selected' : ''; ?>>
                                Calculation from <?php echo date('M j, Y', strtotime($calc['calculation_date'])); ?> - Rs <?php echo number_format($calc['total_zakat'], 2); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="payment_amount">Payment Amount (Rs):</label>
                    <input type="number" step="0.01" id="payment_amount" name="payment_amount" 
                           value="<?php echo $_POST['payment_amount'] ?? ''; ?>" 
                           placeholder="Enter amount in Pakistani Rupees" required>
                </div>

                <div class="form-group">
                    <label for="payment_date">Payment Date:</label>
                    <input type="date" id="payment_date" name="payment_date" 
                           value="<?php echo $_POST['payment_date'] ?? date('Y-m-d'); ?>" required>
                </div>

                <div class="form-group">
                    <label for="payment_method">Payment Method:</label>
                    <select id="payment_method" name="payment_method" required>
                        <option value="">-- Select Method --</option>
                        <option value="Cash" <?php echo ($_POST['payment_method'] ?? '') == 'Cash' ? 'selected' : ''; ?>>Cash</option>
                        <option value="Bank Transfer" <?php echo ($_POST['payment_method'] ?? '') == 'Bank Transfer' ? 'selected' : ''; ?>>Bank Transfer</option>
                        <option value="Credit Card" <?php echo ($_POST['payment_method'] ?? '') == 'Credit Card' ? 'selected' : ''; ?>>Credit Card</option>
                        <option value="Mobile Banking" <?php echo ($_POST['payment_method'] ?? '') == 'Mobile Banking' ? 'selected' : ''; ?>>Mobile Banking</option>
                        <option value="Other" <?php echo ($_POST['payment_method'] ?? '') == 'Other' ? 'selected' : ''; ?>>Other</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="description">Description (Optional):</label>
                    <textarea id="description" name="description" placeholder="Any notes about this payment"><?php echo $_POST['description'] ?? ''; ?></textarea>
                </div>

                <div class="button-group">
                    <button type="submit" class="btn btn-success">💾 Record Payment</button>
                    <a href="payments.php" class="btn btn-primary">📋 View All Payments</a>
                    <a href="dashboard.php" class="btn btn-secondary">📊 Dashboard</a>
                </div>
            </form>
        </div>
    </div>
</body>
</html>