<?php
require_once 'config.php';

// Check if user is logged in
$user_info = getUserInfo($client, $google_oauth);
if (!$user_info) {
    header('Location: login.php');
    exit();
}

$name = $_SESSION['user_name'];
$email = $_SESSION['user_email'];
$picture = $_SESSION['user_picture'] ?? '';

// Get user's Zakat history for summary
$zakat_history = getUserZakatHistory($_SESSION['user_email']);
$total_calculations = count($zakat_history);
$total_zakat = 0;
$recent_calculations = array_slice($zakat_history, 0, 5); // Last 5 calculations

foreach ($zakat_history as $calc) {
    $total_zakat += $calc['total_zakat'];
}

// Safe way to get Zakat summary - check if function exists first
$zakat_summary = [
    'total_zakat' => $total_zakat,
    'total_paid' => 0,
    'remaining_zakat' => $total_zakat
];

// If payment function exists, use it
if (function_exists('getUserZakatSummary')) {
    $payment_summary = getUserZakatSummary($_SESSION['user_email']);
    if ($payment_summary) {
        $zakat_summary = $payment_summary;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Mominist</title>
    <style>
        body {
            padding-top: 80px;
            font-family: 'Arial', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            margin: 0;
            min-height: 100vh;
        }

        .dashboard-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .welcome-section {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            text-align: center;
        }

        .user-avatar {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            margin-bottom: 20px;
            border: 4px solid #4285f4;
            object-fit: cover;
        }

        .user-name {
            color: #333;
            font-size: 2.5em;
            margin-bottom: 10px;
            font-weight: bold;
        }

        .user-email {
            color: #666;
            font-size: 1.2em;
            margin-bottom: 20px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            text-align: center;
            transition: transform 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
        }

        .stat-number {
            font-size: 2.5em;
            font-weight: bold;
            color: #4285f4;
            margin-bottom: 10px;
        }

        .stat-label {
            color: #666;
            font-size: 1.1em;
        }

        .zakat-amount {
            color: #28a745;
        }

        .recent-activity {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }

        .section-title {
            color: #333;
            font-size: 1.8em;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid #f0f0f0;
        }

        .activity-table {
            width: 100%;
            border-collapse: collapse;
        }

        .activity-table th,
        .activity-table td {
            padding: 12px 15px;
            text-align: left;
            border-bottom: 1px solid #f0f0f0;
        }

        .activity-table th {
            background: #f8f9fa;
            font-weight: bold;
            color: #333;
        }

        .zakat-due {
            color: #28a745;
            font-weight: bold;
        }

        .quick-actions {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }

        .action-btn {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            background: white;
            padding: 25px;
            border-radius: 10px;
            text-decoration: none;
            color: #333;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
            text-align: center;
        }

        .action-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.15);
            color: #4285f4;
        }

        .action-icon {
            font-size: 2.5em;
            margin-bottom: 10px;
        }

        .action-label {
            font-size: 1.1em;
            font-weight: bold;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: #666;
        }

        .empty-icon {
            font-size: 4em;
            margin-bottom: 20px;
            opacity: 0.5;
        }

        .welcome-badge {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.9em;
            margin-left: 10px;
        }

        .currency {
            font-family: monospace;
            font-weight: bold;
        }

        @media (max-width: 768px) {
            .dashboard-container {
                padding: 10px;
            }
            
            .user-name {
                font-size: 2em;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .quick-actions {
                grid-template-columns: 1fr;
            }
            
            .activity-table {
                font-size: 14px;
            }
        }
    </style>
</head>
<body>
    <!-- Include Navbar -->
    <?php include 'navbar.php'; ?>

    <div class="dashboard-container">
        <!-- Welcome Section -->
        <div class="welcome-section">
            <?php if (!empty($picture)): ?>
                <img src="<?php echo htmlspecialchars($picture); ?>" alt="Profile Picture" class="user-avatar">
            <?php else: ?>
                <div style="width: 120px; height: 120px; border-radius: 50%; background: #4285f4; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px; color: white; font-size: 3em;">
                    <?php echo strtoupper(substr($name, 0, 1)); ?>
                </div>
            <?php endif; ?>
            
            <h1 class="user-name">
                Welcome, <?php echo htmlspecialchars($name); ?>
                <span class="welcome-badge">✅ Verified</span>
            </h1>
            
            <p class="user-email">📧 <?php echo htmlspecialchars($email); ?></p>
            
            <div style="margin-top: 20px;">
                <p style="color: #666; font-size: 1.1em;">
                    🎉 Welcome to your Mominist dashboard! Track your Zakat calculations and manage your account.
                </p>
            </div>
        </div>

        <!-- Statistics Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?php echo $total_calculations; ?></div>
                <div class="stat-label">Total Calculations</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-number zakat-amount">Rs <?php echo number_format($total_zakat, 2); ?></div>
                <div class="stat-label">Total Zakat Calculated</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-number" style="color: #28a745;">Rs <?php echo number_format($zakat_summary['total_paid'], 2); ?></div>
                <div class="stat-label">Total Paid</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-number" style="color: <?php echo $zakat_summary['remaining_zakat'] > 0 ? '#dc3545' : '#28a745'; ?>;">
                    Rs <?php echo number_format($zakat_summary['remaining_zakat'], 2); ?>
                </div>
                <div class="stat-label">Remaining Zakat</div>
            </div>
        </div>

        <!-- Recent Activity Section -->
        <div class="recent-activity">
            <h2 class="section-title">📈 Recent Zakat Calculations</h2>
            
            <?php if (!empty($recent_calculations)): ?>
                <div style="overflow-x: auto;">
                    <table class="activity-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Gold (g)</th>
                                <th>Silver (g)</th>
                                <th>Cash (Rs)</th>
                                <th>Other Assets (Rs)</th>
                                <th>Zakat Due (Rs)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_calculations as $calculation): ?>
                            <tr>
                                <td><?php echo date('M j, Y', strtotime($calculation['calculation_date'])); ?></td>
                                <td><?php echo number_format($calculation['gold_grams'], 2); ?></td>
                                <td><?php echo number_format($calculation['silver_grams'], 2); ?></td>
                                <td class="currency">Rs <?php echo number_format($calculation['cash_amount'], 2); ?></td>
                                <td class="currency">Rs <?php echo number_format($calculation['other_assets'], 2); ?></td>
                                <td class="zakat-due currency">Rs <?php echo number_format($calculation['total_zakat'], 2); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <?php if ($total_calculations > 5): ?>
                    <div style="text-align: center; margin-top: 20px;">
                        <a href="zakat-history.php" style="background: #4285f4; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block;">
                            View All Calculations
                        </a>
                    </div>
                <?php endif; ?>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-icon">📊</div>
                    <h3>No calculations yet</h3>
                    <p>You haven't made any Zakat calculations yet. Start by creating your first calculation!</p>
                    <a href="zakat-calculator.php" style="background: #28a745; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; margin-top: 15px;">
                        Calculate Zakat Now
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <!-- Quick Actions -->
        <div class="quick-actions">
            <a href="zakat-calculator.php" class="action-btn">
                <div class="action-icon">🧮</div>
                <div class="action-label">Calculate Zakat</div>
                <div style="color: #666; margin-top: 5px; font-size: 0.9em;">New calculation</div>
            </a>
            
            <?php if (function_exists('addZakatPayment')): ?>
            <a href="add-payment.php" class="action-btn">
                <div class="action-icon">💳</div>
                <div class="action-label">Add Payment</div>
                <div style="color: #666; margin-top: 5px; font-size: 0.9em;">Record payment</div>
            </a>
            <?php endif; ?>
            
            <a href="zakat-history.php" class="action-btn">
                <div class="action-icon">📋</div>
                <div class="action-label">Calculations</div>
                <div style="color: #666; margin-top: 5px; font-size: 0.9em;">View history</div>
            </a>
            
            <?php if (function_exists('getUserZakatPayments')): ?>
            <a href="payments.php" class="action-btn">
                <div class="action-icon">💰</div>
                <div class="action-label">Payments</div>
                <div style="color: #666; margin-top: 5px; font-size: 0.9em;">Manage payments</div>
            </a>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>