<?php
require_once 'config.php';

// Check if user is logged in
$user_info = getUserInfo($client, $google_oauth);
if (!$user_info) {
    header('Location: login.php');
    exit();
}

$calculation_id = $_GET['id'] ?? null;
$calculation = null;
$error = '';
$success = '';

// Current market prices
$gold_price_per_gram = 21500;
$silver_price_per_gram = 250;

// Get calculation data
if ($calculation_id) {
    $calculation = getZakatCalculationById($calculation_id, $_SESSION['user_email']);
    if (!$calculation) {
        $error = "Calculation not found or you don't have permission to edit it.";
    }
} else {
    $error = "No calculation ID provided.";
}

// Handle form submission for update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $calculation) {
    // Get form data
    $gold_grams = floatval($_POST['gold_grams'] ?? 0);
    $silver_grams = floatval($_POST['silver_grams'] ?? 0);
    $cash_amount = floatval($_POST['cash_amount'] ?? 0);
    $investments = floatval($_POST['investments'] ?? 0);
    $other_assets = floatval($_POST['other_assets'] ?? 0);
    $liabilities = floatval($_POST['liabilities'] ?? 0);
    
    // Calculate total assets
    $gold_value = $gold_grams * $gold_price_per_gram;
    $silver_value = $silver_grams * $silver_price_per_gram;
    $total_assets = $gold_value + $silver_value + $cash_amount + $investments + $other_assets;
    
    // Calculate net assets
    $net_assets = $total_assets - $liabilities;
    
    // Nisab threshold
    $nisab = 612.36 * $silver_price_per_gram;
    
    // Calculate Zakat (2.5% of net assets if above nisab)
    $total_zakat = 0;
    if ($net_assets >= $nisab) {
        $total_zakat = $net_assets * 0.025;
    }
    
    $calculation_data = [
        'gold_grams' => $gold_grams,
        'silver_grams' => $silver_grams,
        'cash_amount' => $cash_amount,
        'investments' => $investments,
        'other_assets' => $other_assets,
        'liabilities' => $liabilities,
        'total_zakat' => $total_zakat
    ];
    
    // Update calculation
    if (updateZakatCalculation($calculation_id, $_SESSION['user_email'], $calculation_data)) {
        $success = "Calculation updated successfully!";
        // Refresh calculation data
        $calculation = getZakatCalculationById($calculation_id, $_SESSION['user_email']);
    } else {
        $error = "Failed to update calculation. Please try again.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Calculation - Mominist</title>
    <style>
        body {
            padding-top: 80px;
            font-family: Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            margin: 0;
            min-height: 100vh;
        }

        .edit-container {
            max-width: 800px;
            margin: 20px auto;
            padding: 0 20px;
        }

        .edit-card {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
            margin-bottom: 20px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #333;
        }

        .form-group input {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
        }

        .input-hint {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            font-weight: bold;
            text-decoration: none;
            display: inline-block;
            text-align: center;
            transition: all 0.3s;
        }

        .btn-primary {
            background: #4285f4;
            color: white;
        }

        .btn-success {
            background: #28a745;
            color: white;
        }

        .btn-danger {
            background: #dc3545;
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn:hover {
            opacity: 0.9;
            transform: translateY(-2px);
        }

        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap;
        }

        .alert {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .calculation-info {
            background: #e8f5e8;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border-left: 4px solid #28a745;
        }

        .price-info {
            background: #e2e3e5;
            color: #383d41;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 15px;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <!-- Include Navbar -->
    <?php include 'navbar.php'; ?>

    <div class="edit-container">
        <div class="edit-card">
            <h1 style="text-align: center; color: #333; margin-bottom: 30px;">Edit Zakat Calculation</h1>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <?php echo $success; ?>
                </div>
            <?php endif; ?>

            <?php if ($calculation): ?>
                <div class="calculation-info">
                    <strong>Original Calculation Date:</strong> 
                    <?php echo date('F j, Y g:i A', strtotime($calculation['calculation_date'])); ?>
                </div>

                <div class="price-info">
                    💰 <strong>Current Prices:</strong> Gold: Rs <?php echo number_format($gold_price_per_gram); ?>/gram | 
                    Silver: Rs <?php echo number_format($silver_price_per_gram); ?>/gram
                </div>

                <form method="POST" action="">
                    <div class="form-group">
                        <label for="gold_grams">Gold (grams):</label>
                        <input type="number" step="0.01" id="gold_grams" name="gold_grams" 
                               value="<?php echo htmlspecialchars($calculation['gold_grams']); ?>" 
                               placeholder="Enter gold in grams" required>
                        <div class="input-hint">Value: Rs <?php echo number_format($gold_price_per_gram); ?> per gram</div>
                    </div>

                    <div class="form-group">
                        <label for="silver_grams">Silver (grams):</label>
                        <input type="number" step="0.01" id="silver_grams" name="silver_grams" 
                               value="<?php echo htmlspecialchars($calculation['silver_grams']); ?>" 
                               placeholder="Enter silver in grams" required>
                        <div class="input-hint">Value: Rs <?php echo number_format($silver_price_per_gram); ?> per gram</div>
                    </div>

                    <div class="form-group">
                        <label for="cash_amount">Cash & Bank Balance (Rs):</label>
                        <input type="number" step="0.01" id="cash_amount" name="cash_amount" 
                               value="<?php echo htmlspecialchars($calculation['cash_amount']); ?>" 
                               placeholder="Enter amount in Pakistani Rupees" required>
                    </div>

                    <div class="form-group">
                        <label for="investments">Investments & Stocks (Rs):</label>
                        <input type="number" step="0.01" id="investments" name="investments" 
                               value="<?php echo htmlspecialchars($calculation['investments']); ?>" 
                               placeholder="Enter investments value in Rupees" required>
                    </div>

                    <div class="form-group">
                        <label for="other_assets">Other Assets (Rs):</label>
                        <input type="number" step="0.01" id="other_assets" name="other_assets" 
                               value="<?php echo htmlspecialchars($calculation['other_assets']); ?>" 
                               placeholder="Enter other assets value in Rupees" required>
                        <div class="input-hint">Property, Business, Vehicles, etc.</div>
                    </div>

                    <div class="form-group">
                        <label for="liabilities">Liabilities & Debts (Rs):</label>
                        <input type="number" step="0.01" id="liabilities" name="liabilities" 
                               value="<?php echo htmlspecialchars($calculation['liabilities']); ?>" 
                               placeholder="Enter liabilities amount in Rupees" required>
                        <div class="input-hint">Loans, Credit Card Debt, etc.</div>
                    </div>

                    <div class="button-group">
                        <button type="submit" class="btn btn-success">💾 Update Calculation</button>
                        <a href="zakat-history.php" class="btn btn-secondary">📋 Back to History</a>
                        <a href="dashboard.php" class="btn btn-primary">📊 Dashboard</a>
                        <a href="delete-calculation.php?id=<?php echo $calculation_id; ?>" 
                           class="btn btn-danger" 
                           onclick="return confirm('Are you sure you want to delete this calculation? This action cannot be undone.')">
                           🗑️ Delete
                        </a>
                    </div>
                </form>
            <?php else: ?>
                <div style="text-align: center; padding: 40px;">
                    <p>Calculation not found or you don't have permission to edit it.</p>
                    <a href="zakat-history.php" class="btn btn-primary">Back to History</a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>