<?php
require_once 'config.php';

// Get user info for navbar
getUserInfo($client, $google_oauth);

$isLoggedIn = isset($_SESSION['user_email']);
$calculation_result = null;
$save_success = false;
$db_error = false;
$error = '';
$success = '';

// Function to fetch metal prices from free Gold API
function fetchMetalPrices() {
    $metals = ['XAU', 'XAG'];
    $prices = [];
    
    foreach ($metals as $metal) {
        $url = "https://api.gold-api.com/price/{$metal}";
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_USERAGENT => 'Mominist Zakat Calculator'
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200 && $response) {
            $data = json_decode($response, true);
            if (isset($data['price'])) {
                $prices[$metal] = $data['price'];
            }
        }
    }
    
    return $prices;
}

// Get current metal prices
function getCurrentMetalPrices() {
    // Try to fetch from API first
    $api_prices = fetchMetalPrices();
    
    if (!empty($api_prices)) {
        // API call successful
        $gold_price_per_gram = isset($api_prices['XAU']) ? $api_prices['XAU'] : 21500;
        $silver_price_per_gram = isset($api_prices['XAG']) ? $api_prices['XAG'] : 250;
        $source = 'api';
    } else {
        // Fallback to cached/default prices
        $gold_price_per_gram = 21500;
        $silver_price_per_gram = 250;
        $source = 'cached';
    }
    
    // Calculate tola prices (1 tola = 11.66 grams)
    $gold_price_per_tola = $gold_price_per_gram * 11.66;
    $silver_price_per_tola = $silver_price_per_gram * 11.66;
    
    return [
        'gold_gram' => $gold_price_per_gram,
        'gold_tola' => $gold_price_per_tola,
        'silver_gram' => $silver_price_per_gram,
        'silver_tola' => $silver_price_per_tola,
        'source' => $source
    ];
}

// Get current metal prices
$metal_prices = getCurrentMetalPrices();
$gold_price_per_gram = $metal_prices['gold_gram'];
$gold_price_per_tola = $metal_prices['gold_tola'];
$silver_price_per_gram = $metal_prices['silver_gram'];
$silver_price_per_tola = $metal_prices['silver_tola'];
$price_source = $metal_prices['source'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get form data with unit conversion
    $gold_unit = $_POST['gold_unit'] ?? 'gram';
    $gold_value = floatval($_POST['gold_value'] ?? 0);
    $gold_grams = $gold_unit === 'tola' ? $gold_value * 11.66 : $gold_value;
    
    $silver_unit = $_POST['silver_unit'] ?? 'gram';
    $silver_value = floatval($_POST['silver_value'] ?? 0);
    $silver_grams = $silver_unit === 'tola' ? $silver_value * 11.66 : $silver_value;
    
    $cash_amount = floatval($_POST['cash_amount'] ?? 0);
    $investments = floatval($_POST['investments'] ?? 0);
    $other_assets = floatval($_POST['other_assets'] ?? 0);
    $liabilities = floatval($_POST['liabilities'] ?? 0);
    
    // Validate inputs
    if ($gold_value < 0 || $silver_value < 0 || $cash_amount < 0 || $investments < 0 || $other_assets < 0 || $liabilities < 0) {
        $error = "Please enter positive values for all fields.";
    } else {
        // Calculate total assets
        $gold_value_rs = $gold_grams * $gold_price_per_gram;
        $silver_value_rs = $silver_grams * $silver_price_per_gram;
        $total_assets = $gold_value_rs + $silver_value_rs + $cash_amount + $investments + $other_assets;
        
        // Calculate net assets
        $net_assets = $total_assets - $liabilities;
        
        // Nisab threshold (current silver nisab: 612.36 grams * silver price)
        $nisab = 612.36 * $silver_price_per_gram;
        
        // Calculate Zakat (2.5% of net assets if above nisab)
        $total_zakat = 0;
        if ($net_assets >= $nisab) {
            $total_zakat = $net_assets * 0.025;
        }
        
        $calculation_result = [
            'gold_grams' => $gold_grams,
            'gold_value' => $gold_value,
            'gold_unit' => $gold_unit,
            'silver_grams' => $silver_grams,
            'silver_value' => $silver_value,
            'silver_unit' => $silver_unit,
            'cash_amount' => $cash_amount,
            'investments' => $investments,
            'other_assets' => $other_assets,
            'liabilities' => $liabilities,
            'gold_value_rs' => $gold_value_rs,
            'silver_value_rs' => $silver_value_rs,
            'total_assets' => $total_assets,
            'net_assets' => $net_assets,
            'nisab' => $nisab,
            'total_zakat' => $total_zakat,
            'meets_nisab' => $net_assets >= $nisab
        ];
        
        // Save calculation if user is logged in
        if ($isLoggedIn && $total_zakat > 0) {
            $save_data = [
                'gold_grams' => $gold_grams,
                'silver_grams' => $silver_grams,
                'cash_amount' => $cash_amount,
                'investments' => $investments,
                'other_assets' => $other_assets,
                'liabilities' => $liabilities,
                'total_zakat' => $total_zakat
            ];
            
            $save_result = saveZakatCalculation($_SESSION['user_email'], $_SESSION['user_name'], $save_data);
            $save_success = ($save_result !== false);
            $db_error = ($save_result === false);
            
            if ($save_success) {
                $success = "Calculation saved successfully!";
            } elseif ($db_error) {
                $error = "Calculation completed but couldn't save to database.";
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Zakat Calculator - Mominist</title>
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-light">
    <!-- Include Navbar -->
    <?php include 'navbar.php'; ?>

    <div class="container py-4">
        <div class="row justify-content-center">
            <div class="col-lg-10">
                <!-- Calculator Card -->
                <div class="card shadow-lg border-0">
                    <!-- Header -->
                    <div class="card-header bg-success text-white py-4">
                        <div class="text-center">
                            <h1 class="display-5 fw-bold mb-3">
                                <i class="fas fa-calculator me-3"></i>Zakat Calculator
                            </h1>
                            <p class="lead mb-0">Calculate your Zakat accurately with real-time market prices</p>
                        </div>
                    </div>

                    <div class="card-body p-4 p-md-5">
                        <!-- Current Prices -->
                        <div class="card bg-primary text-white mb-4">
                            <div class="card-body">
                                <div class="row text-center">
                                    <div class="col-md-6 mb-4 mb-md-0">
                                        <h5 class="fw-bold mb-3">
                                            <i class="fas fa-crown me-2"></i>Gold Prices 
                                            <span class="badge bg-<?php echo $price_source === 'api' ? 'success' : 'warning'; ?> ms-2">
                                                <i class="fas <?php echo $price_source === 'api' ? 'fa-bolt' : 'fa-database'; ?> me-1"></i>
                                                <?php echo $price_source === 'api' ? 'LIVE' : 'CACHED'; ?>
                                            </span>
                                        </h5>
                                        <div class="h3 fw-bold mb-2">Rs <?php echo number_format($gold_price_per_gram, 2); ?>/gram</div>
                                        <div class="h5 opacity-75">Rs <?php echo number_format($gold_price_per_tola, 2); ?>/tola</div>
                                    </div>
                                    <div class="col-md-6">
                                        <h5 class="fw-bold mb-3">
                                            <i class="fas fa-gem me-2"></i>Silver Prices
                                            <span class="badge bg-<?php echo $price_source === 'api' ? 'success' : 'warning'; ?> ms-2">
                                                <i class="fas <?php echo $price_source === 'api' ? 'fa-bolt' : 'fa-database'; ?> me-1"></i>
                                                <?php echo $price_source === 'api' ? 'LIVE' : 'CACHED'; ?>
                                            </span>
                                        </h5>
                                        <div class="h3 fw-bold mb-2">Rs <?php echo number_format($silver_price_per_gram, 2); ?>/gram</div>
                                        <div class="h5 opacity-75">Rs <?php echo number_format($silver_price_per_tola, 2); ?>/tola</div>
                                    </div>
                                </div>
                                <?php if ($price_source === 'api'): ?>
                                    <div class="text-center mt-3 pt-3 border-top border-white border-opacity-25">
                                        <small><i class="fas fa-sync-alt me-1"></i>Live prices from GoldAPI</small>
                                    </div>
                                <?php else: ?>
                                    <div class="text-center mt-3 pt-3 border-top border-white border-opacity-25">
                                        <small><i class="fas fa-exclamation-triangle me-1"></i>Using cached prices - API unavailable</small>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Alerts -->
                        <?php if ($error): ?>
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                <?php echo $error; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <?php if ($success): ?>
                            <div class="alert alert-success alert-dismissible fade show" role="alert">
                                <i class="fas fa-check-circle me-2"></i>
                                <?php echo $success; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <!-- Calculator Form -->
                        <form method="POST" action="">
                            <div class="row">
                                <!-- Gold Section -->
                                <div class="col-lg-6 mb-4">
                                    <div class="card h-100 border-0 shadow-sm">
                                        <div class="card-header bg-warning text-dark">
                                            <h5 class="card-title mb-0">
                                                <i class="fas fa-crown me-2"></i>Gold Assets
                                            </h5>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <label class="form-label fw-bold">Gold Quantity</label>
                                                <div class="input-group">
                                                    <input type="number" step="0.01" class="form-control form-control-lg" 
                                                           name="gold_value" value="<?php echo $_POST['gold_value'] ?? ''; ?>" 
                                                           placeholder="Enter quantity" >
                                                    <select class="form-select form-select-lg" name="gold_unit" style="min-width: 120px;">
                                                        <option value="gram" <?php echo ($_POST['gold_unit'] ?? 'gram') == 'gram' ? 'selected' : ''; ?>>Grams</option>
                                                        <option value="tola" <?php echo ($_POST['gold_unit'] ?? 'gram') == 'tola' ? 'selected' : ''; ?>>Tola</option>
                                                    </select>
                                                </div>
                                                <div class="form-text text-muted mt-2">
                                                    <i class="fas fa-info-circle me-1"></i>1 Tola = 11.66 Grams
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Silver Section -->
                                <div class="col-lg-6 mb-4">
                                    <div class="card h-100 border-0 shadow-sm">
                                        <div class="card-header bg-secondary text-white">
                                            <h5 class="card-title mb-0">
                                                <i class="fas fa-gem me-2"></i>Silver Assets
                                            </h5>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-3">
                                                <label class="form-label fw-bold">Silver Quantity</label>
                                                <div class="input-group">
                                                    <input type="number" step="0.01" class="form-control form-control-lg" 
                                                           name="silver_value" value="<?php echo $_POST['silver_value'] ?? ''; ?>" 
                                                           placeholder="Enter quantity" >
                                                    <select class="form-select form-select-lg" name="silver_unit" style="min-width: 120px;">
                                                        <option value="gram" <?php echo ($_POST['silver_unit'] ?? 'gram') == 'gram' ? 'selected' : ''; ?>>Grams</option>
                                                        <option value="tola" <?php echo ($_POST['silver_unit'] ?? 'gram') == 'tola' ? 'selected' : ''; ?>>Tola</option>
                                                    </select>
                                                </div>
                                                <div class="form-text text-muted mt-2">
                                                    <i class="fas fa-info-circle me-1"></i>1 Tola = 11.66 Grams
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Financial Assets -->
                            <div class="card border-0 shadow-sm mb-4">
                                <div class="card-header bg-success text-white">
                                    <h5 class="card-title mb-0">
                                        <i class="fas fa-money-bill-wave me-2"></i>Financial Assets & Liabilities
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-bold">Cash & Bank Balance</label>
                                            <div class="input-group">
                                                <span class="input-group-text bg-light fw-bold">Rs</span>
                                                <input type="number" step="0.01" class="form-control" 
                                                       name="cash_amount" value="<?php echo $_POST['cash_amount'] ?? ''; ?>" 
                                                       placeholder="Enter amount in Pakistani Rupees">
                                            </div>
                                        </div>

                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-bold">Investments & Stocks</label>
                                            <div class="input-group">
                                                <span class="input-group-text bg-light fw-bold">Rs</span>
                                                <input type="number" step="0.01" class="form-control" 
                                                       name="investments" value="<?php echo $_POST['investments'] ?? ''; ?>" 
                                                       placeholder="Enter investments value">
                                            </div>
                                        </div>

                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-bold">Other Assets</label>
                                            <div class="input-group">
                                                <span class="input-group-text bg-light fw-bold">Rs</span>
                                                <input type="number" step="0.01" class="form-control" 
                                                       name="other_assets" value="<?php echo $_POST['other_assets'] ?? ''; ?>" 
                                                       placeholder="Property, Business, Vehicles, etc.">
                                            </div>
                                        </div>

                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-bold">Liabilities & Debts</label>
                                            <div class="input-group">
                                                <span class="input-group-text bg-light fw-bold">Rs</span>
                                                <input type="number" step="0.01" class="form-control" 
                                                       name="liabilities" value="<?php echo $_POST['liabilities'] ?? ''; ?>" 
                                                       placeholder="Loans, Credit Cards, Mortgages, etc.">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Calculate Button -->
                            <div class="text-center mt-4">
                                <button type="submit" class="btn btn-success btn-lg px-5 py-3">
                                    <i class="fas fa-calculator me-2"></i>Calculate Zakat
                                </button>
                            </div>
                        </form>

                        <!-- Results Section -->
                        <?php if ($calculation_result): ?>
                            <div class="mt-5 pt-4 border-top">
                                <h4 class="text-center mb-4">
                                    <i class="fas fa-chart-bar text-primary me-2"></i>Calculation Results
                                </h4>
                                
                                <!-- Assets Breakdown -->
                                <div class="card border-0 shadow-sm mb-4">
                                    <div class="card-header bg-info text-white">
                                        <h5 class="card-title mb-0">
                                            <i class="fas fa-chart-pie me-2"></i>Assets Breakdown
                                        </h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="row g-3">
                                            <?php if ($calculation_result['gold_grams'] > 0): ?>
                                            <div class="col-md-6">
                                                <div class="card border-warning">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <h6 class="card-title text-warning">
                                                                    <i class="fas fa-crown me-2"></i>Gold
                                                                </h6>
                                                                <p class="card-text text-muted mb-0">
                                                                    <?php echo number_format($calculation_result['gold_value'], 2); ?> <?php echo $calculation_result['gold_unit']; ?>
                                                                </p>
                                                            </div>
                                                            <div class="h5 mb-0 fw-bold text-warning">
                                                                Rs <?php echo number_format($calculation_result['gold_value_rs'], 2); ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <?php if ($calculation_result['silver_grams'] > 0): ?>
                                            <div class="col-md-6">
                                                <div class="card border-secondary">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <h6 class="card-title text-secondary">
                                                                    <i class="fas fa-gem me-2"></i>Silver
                                                                </h6>
                                                                <p class="card-text text-muted mb-0">
                                                                    <?php echo number_format($calculation_result['silver_value'], 2); ?> <?php echo $calculation_result['silver_unit']; ?>
                                                                </p>
                                                            </div>
                                                            <div class="h5 mb-0 fw-bold text-secondary">
                                                                Rs <?php echo number_format($calculation_result['silver_value_rs'], 2); ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <?php if ($calculation_result['cash_amount'] > 0): ?>
                                            <div class="col-md-6">
                                                <div class="card border-success">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <h6 class="card-title text-success">
                                                                    <i class="fas fa-money-bill me-2"></i>Cash & Bank
                                                                </h6>
                                                                <p class="card-text text-muted mb-0">Liquid Assets</p>
                                                            </div>
                                                            <div class="h5 mb-0 fw-bold text-success">
                                                                Rs <?php echo number_format($calculation_result['cash_amount'], 2); ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <?php if ($calculation_result['investments'] > 0): ?>
                                            <div class="col-md-6">
                                                <div class="card border-primary">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <h6 class="card-title text-primary">
                                                                    <i class="fas fa-chart-line me-2"></i>Investments
                                                                </h6>
                                                                <p class="card-text text-muted mb-0">Stocks & Bonds</p>
                                                            </div>
                                                            <div class="h5 mb-0 fw-bold text-primary">
                                                                Rs <?php echo number_format($calculation_result['investments'], 2); ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <?php if ($calculation_result['other_assets'] > 0): ?>
                                            <div class="col-md-6">
                                                <div class="card border-info">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <div>
                                                                <h6 class="card-title text-info">
                                                                    <i class="fas fa-building me-2"></i>Other Assets
                                                                </h6>
                                                                <p class="card-text text-muted mb-0">Property & Business</p>
                                                            </div>
                                                            <div class="h5 mb-0 fw-bold text-info">
                                                                Rs <?php echo number_format($calculation_result['other_assets'], 2); ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endif; ?>
                                        </div>

                                        <!-- Totals -->
                                        <div class="row mt-4">
                                            <div class="col-12">
                                                <div class="card bg-success text-white">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <h5 class="mb-0">Total Assets</h5>
                                                            <h4 class="mb-0">Rs <?php echo number_format($calculation_result['total_assets'], 2); ?></h4>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <?php if ($calculation_result['liabilities'] > 0): ?>
                                            <div class="col-12 mt-3">
                                                <div class="card bg-danger text-white">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <h5 class="mb-0">Liabilities & Debts</h5>
                                                            <h4 class="mb-0">- Rs <?php echo number_format($calculation_result['liabilities'], 2); ?></h4>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <div class="col-12 mt-3">
                                                <div class="card bg-primary text-white">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <h4 class="mb-0">Net Assets</h4>
                                                            <h3 class="mb-0">Rs <?php echo number_format($calculation_result['net_assets'], 2); ?></h3>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Nisab Check -->
                                <div class="card border-0 shadow-sm mb-4">
                                    <div class="card-header bg-warning text-dark">
                                        <h5 class="card-title mb-0">
                                            <i class="fas fa-scale-balanced me-2"></i>Nisab Eligibility Check
                                        </h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="row text-center">
                                            <div class="col-md-6 mb-4">
                                                <div class="card border-success">
                                                    <div class="card-body">
                                                        <h6 class="card-title">Your Net Assets</h6>
                                                        <div class="h3 fw-bold text-success">Rs <?php echo number_format($calculation_result['net_assets'], 2); ?></div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-md-6 mb-4">
                                                <div class="card border-info">
                                                    <div class="card-body">
                                                        <h6 class="card-title">Nisab Threshold</h6>
                                                        <div class="h3 fw-bold text-info">Rs <?php echo number_format($calculation_result['nisab'], 2); ?></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="text-center mt-3">
                                            <?php if ($calculation_result['meets_nisab']): ?>
                                                <div class="alert alert-success">
                                                    <h5 class="mb-0">
                                                        <i class="fas fa-check-circle me-2"></i>You are eligible to pay Zakat
                                                    </h5>
                                                </div>
                                            <?php else: ?>
                                                <div class="alert alert-warning">
                                                    <h5 class="mb-0">
                                                        <i class="fas fa-info-circle me-2"></i>Your assets are below nisab threshold
                                                    </h5>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>

                                <!-- Zakat Result -->
                                <?php if ($calculation_result['meets_nisab']): ?>
                                    <div class="card border-0 shadow-lg">
                                        <div class="card-header bg-gradient bg-warning text-dark text-center py-4">
                                            <h3 class="card-title mb-0">
                                                <i class="fas fa-check-circle me-2"></i>Zakat Calculation Complete
                                            </h3>
                                        </div>
                                        <div class="card-body text-center py-5">
                                            <div class="display-4 fw-bold text-success mb-4">
                                                Rs <?php echo number_format($calculation_result['total_zakat'], 2); ?>
                                            </div>
                                            <p class="h4 text-muted mb-4">Your Zakat Due (2.5% of net assets above nisab)</p>
                                            
                                            <?php if ($isLoggedIn): ?>
                                                <?php if ($save_success): ?>
                                                    <div class="alert alert-success">
                                                        <i class="fas fa-save me-2"></i>Calculation saved to your history!
                                                    </div>
                                                <?php elseif ($db_error): ?>
                                                    <div class="alert alert-warning">
                                                        <i class="fas fa-exclamation-triangle me-2"></i>Calculation completed but couldn't save to database
                                                    </div>
                                                <?php else: ?>
                                                    <div class="alert alert-success">
                                                        <i class="fas fa-check me-2"></i>Calculation automatically saved to your account
                                                    </div>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <div class="card border-danger">
                                                    <div class="card-body">
                                                        <h5 class="card-title text-danger">
                                                            <i class="fas fa-lock me-2"></i>Save Your Calculation
                                                        </h5>
                                                        <p class="card-text">Login to save this calculation and track your Zakat payments over time</p>
                                                        <a href="login.php" class="btn btn-danger btn-lg">
                                                            <i class="fab fa-google me-2"></i>Login with Gmail to Save
                                                        </a>
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-warning text-center py-4">
                                        <h4 class="alert-heading mb-3">
                                            <i class="fas fa-info-circle me-2"></i>Not Required to Pay Zakat
                                        </h4>
                                        <p class="h5 mb-0">Your net assets are below the nisab threshold. You are not required to pay Zakat at this time.</p>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap 5 JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Auto-format numbers on blur
        document.addEventListener('DOMContentLoaded', function() {
            const numberInputs = document.querySelectorAll('input[type="number"]');
            numberInputs.forEach(input => {
                input.addEventListener('blur', function() {
                    if (this.value) {
                        this.value = parseFloat(this.value).toFixed(2);
                    }
                });
            });
        });
    </script>
</body>
</html>